/*
 * bztclLiteCmd.c --
 * Copyright (c) 1997 Clif Flynt
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL Clif Flynt BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Clif Flynt SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND Clif Flynt HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
*/
/* Include the usual suspects.						*/

#include "stdlib.h"
#include "ctype.h"
#include "bztclLiteInt.h"

#define BZTCLLITECMD_REVISION "$Revision: 1.4 $"

/*
* Define the sub commands                          
*
* These strings define the subcommands that the bztclLite command 
* supports.
*
* To add a new subcommand, add the new subcommand string, 
* #define and entry in cmdDefinition
*
* Note: Order is important.  
*
* These are the subcommands that will be recognized
*
*/

static char *subcommands[] = {
	"compress", "uncompress", "readFile", "writeFile", NULL};

/*
* These #defines define the positions of the subcommands.
* You can use enums if you are certain your compiler will provide the
* same numbers as this.
*/

#define M_compress 0
#define M_uncompress    1
#define M_readFile      2
#define M_writeFile     3
#define M_bogus         4

                
typedef struct ByteArray {
    int used;                   /* The number of bytes used in the byte
                                 * array. */
    int allocated;              /* The amount of space actually allocated
                                 * minus 1 byte. */
    unsigned char bytes[4];     /* The array of bytes.  The actual size of
                                 * this field depends on the 'allocated' field
                                 * above. */
} ByteArray;
 

/*
* The cmdDefinition structure describes the minimum and maximum number
*  of expected arguments for the subcommand (including cmd and subcommand
*  names), and a usage message to return if the argument
*  count is outside the expected range.
*/

typedef struct cmd_Def {
   char *usage;
   int  minArgCnt;
   int  maxArgCnt;
   } cmdDefinition;

static cmdDefinition definitions[] = {
    {"compress string", 3 , 3},
    {"uncompress string", 3 , 3},
    {"readFile fileName", 3 , 3},
    {"writeFile fileName string", 4 , 4},
    {"bogus bogusval", 3,3}
};

/* If bztclLiteDebugPrint != 0, then debugprt will print debugging info */
/*   This value is set with the default subcommand function debug   */

int bztclLiteDebugPrint = 0;

/* ----- C++ Comment
// These are the segments that need to be declared in the "C" namespace
// These function names and data will be accessed by programs in the
//   Tcl "C" language library.
*/

/*
extern "C" {
*/

/*------------------------------------------------------------------------
* CmdReturn *bztclLite_compressCmd ()--
*    Demonstrates creating a hash entry.
*    Creates a hash entry for Key, with value String
* Arguments
*   objv[0]:	"dummy"
*   objv[1]:	"create"
*   objv[2]:	hash Key
*   objv[3]:	String
* 
* Results
*   Creates a new hash entry.  Sets error if entry already exists.
*
* Side Effects:
*   None
------------------------------------------------------------------------*/
CmdReturn *bztclLite_compressCmd (ClientData info, 
			 Tcl_Interp *interp, 
			 int objc, 
			 Tcl_Obj * CONST *objv) {
    CmdReturn *returnStructPtr;
    unsigned char *returnCharPtr;
    Tcl_Obj *returnObjPtr;

    char *inbuf, *outbuf;
    bz_stream *bz1;
    int byteCount, out, charCount;
    
    byteCount = 0;

    /*
     * Print that the function was called for debugging
     */

    debugprt("bztclLite_compressCmd called with %d args\n", objc);

    /* 
     * Allocate the space and initialize the return structure.
     */

    returnStructPtr = (CmdReturn *) malloc(sizeof (CmdReturn));
    returnStructPtr->status = TCL_OK;
    returnObjPtr    = NULL;


    inbuf = Tcl_GetStringFromObj(objv[2], &charCount);
    charCount *= 2;
    outbuf = (char *)ckalloc(charCount);
    returnCharPtr = (unsigned char *)malloc(charCount);

    bz1 = (bz_stream *) malloc(sizeof(bz_stream));
    bz1->bzalloc = NULL;
    bz1->bzfree  = NULL;
    bz1->opaque  = NULL;
    bz1->next_in = inbuf;
    bz1->next_out = outbuf;
    bz1->avail_in = strlen(inbuf);
    bz1->avail_out= charCount;
    bz1->total_out_lo32 = 0;
    bz1->total_out_hi32 = 0;
  
  out = BZ2_bzCompressInit (bz1, 9, 0, 30);
  debugprt("out0: %d IN: %d, %d OUT: %d\n", out, bz1->avail_in, bz1->total_in_lo32, bz1->total_out_lo32);
  out = -1;
  while (out != BZ_STREAM_END) {
    out = BZ2_bzCompress(bz1, BZ_FINISH);
    debugprt("out2: %d IN: %d, %d OUT: %d\n", out, bz1->avail_in, bz1->total_in_lo32, bz1->total_out_lo32);
    memcpy(&returnCharPtr[byteCount], outbuf, bz1->total_out_lo32);
    byteCount += bz1->total_out_lo32;
    if (out != BZ_STREAM_END) {
      returnCharPtr = (unsigned char *)realloc(returnCharPtr, byteCount + 10000);
    }
  }
  
  if (returnCharPtr != NULL) {
    returnCharPtr[byteCount] = 0;
  }

    out = BZ2_bzCompressEnd(bz1);
    debugprt("out3: %d IN: %d, %d OUT: %d\n", out, bz1->avail_in, bz1->total_in_lo32, bz1->total_out_lo32);
  

done:

    if ((returnObjPtr == NULL) && (returnCharPtr != NULL)) {
        returnObjPtr = Tcl_NewByteArrayObj(returnCharPtr, byteCount);
     }
 
                             
    returnStructPtr->object = returnObjPtr;
    if (returnCharPtr != NULL) {free(returnCharPtr);}
 
    free(bz1);
    ckfree (outbuf);
    return returnStructPtr;
}

/*------------------------------------------------------------------------
* CmdReturn *bztclLite_uncompressCmd ()--
*    Demonstrates creating a hash entry.
*    Creates a hash entry for Key, with value String
* Arguments
*   objv[0]:	"dummy"
*   objv[1]:	"create"
*   objv[2]:	hash Key
*   objv[3]:	String
* 
* Results
*   Creates a new hash entry.  Sets error if entry already exists.
*
* Side Effects:
*   None
------------------------------------------------------------------------*/
CmdReturn *bztclLite_uncompressCmd (ClientData info, 
			 Tcl_Interp *interp, 
			 int objc, 
			 Tcl_Obj * CONST *objv) {
    CmdReturn *returnStructPtr;
    char *returnCharPtr;
    Tcl_Obj *returnObjPtr;

    unsigned char *inbuf;
    int byteCount, out;
    int comprCount;
    int decomprCount = 0;
    ByteArray *byteArrayPtr;

    /*
     * Print that the function was called for debugging
     */

    debugprt("bztclLite_uncompressCmd called with %d args\n", objc);

    /* 
     * Allocate the space and initialize the return structure.
     */

    returnStructPtr = (CmdReturn *) malloc(sizeof (CmdReturn));
    returnStructPtr->status = TCL_OK;
    returnObjPtr    = NULL;

    if (0 == strcmp("string", objv[2]->typePtr->name)) {
      inbuf = (unsigned char *) Tcl_GetStringFromObj(objv[2], &comprCount);
    } else {
      inbuf = Tcl_GetByteArrayFromObj(objv[2], &comprCount);
    }
// Not work at all.
//    inbuf = Tcl_GetUnicodeFromObj(objv[2], &comprCount);

    returnCharPtr = (char *) bztclLite_DecompressBuffer (inbuf, comprCount, &decomprCount);

done:
    if ((returnObjPtr == NULL) && (returnCharPtr != NULL)) {
        returnObjPtr = Tcl_NewStringObj(returnCharPtr, decomprCount);
    }
                            
    returnStructPtr->object = returnObjPtr;
    if (returnCharPtr != NULL) {ckfree(returnCharPtr);}

fail:    
    return returnStructPtr;
}

/*------------------------------------------------------------------------
* unsigned char *bztclLite_DecompressBuffer--
*   Decompress compressed data into an undefined buffer, return the buffer
*    which will need to be ckfreed later.
* Arguments:
*   compressed		Compressed Data
*   comprLen		Number of bytes to decompress
*   decomprCount	Pointer to int to receive length of returned data
* Results:
*   Returns a pointer to a buffer containing N returned bytes.
* 
* Side Effects:
*   ckallocs a buffer.
*
------------------------------------------------------------------------*/
unsigned char *bztclLite_DecompressBuffer (
      unsigned char *compressed, int comprLen, int *decomprCount) {
    char *outbuf, *returnCharPtr;
    bz_stream *bz1;
    int byteCount, out;
    int charCount;
    
    charCount = comprLen;
    outbuf = (char *)ckalloc(10000);
    returnCharPtr = (char *)ckalloc(10000);

    byteCount = 0;

    bz1 = (bz_stream *) malloc(sizeof(bz_stream));
    bz1->bzalloc = NULL;
    bz1->bzfree  = NULL;
    bz1->opaque  = NULL;
    bz1->next_in = (char *)  compressed;
    bz1->next_out = outbuf;
    bz1->avail_out = 10000;
    bz1->avail_in = charCount;
    bz1->total_out_lo32 = 0;
    bz1->total_out_hi32 = 0;
    bz1->total_in_lo32 = 0;
    bz1->total_in_hi32 = 0;

  out = BZ2_bzDecompressInit (bz1, 0, 0);
  debugprt("outU0: %d nextin: %xx, AVAILin: %d TOTIN: %d NextOut: %xx AVOUT %d TOTOUT: %d  byteCount: %d\n", 
        out, bz1->next_in, bz1->avail_in, bz1->total_in_lo32, 
	bz1->next_out, bz1->avail_out, bz1->total_out_lo32,
	byteCount);

  while ((out == BZ_OK)) {
    out = BZ2_bzDecompress(bz1);
    debugprt("outU2: %d nextin: %xx, AVAILin: %d TOTIN: %d NextOut: %xx AVOUT %d TOTOUT: %d  byteCount: %d\n", 
        out, bz1->next_in, bz1->avail_in, bz1->total_in_lo32, 
	bz1->next_out, bz1->avail_out, bz1->total_out_lo32,
	byteCount);
    
    if (out < 0) {
      ckfree(returnCharPtr);
      returnCharPtr= NULL;
      byteCount = 0;
      goto fail;
    }

    memcpy(&returnCharPtr[byteCount], outbuf, 10000-bz1->avail_out);
    byteCount = bz1->total_out_lo32;

    if (out != BZ_STREAM_END) {
      returnCharPtr = (char *)ckrealloc(returnCharPtr, byteCount + 10000);
      bz1->next_out = outbuf;
      bz1->avail_out = 10000;
    }
  }
  
    out = BZ2_bzDecompressEnd(bz1);
    debugprt("outU3: %d IN: %d, %d OUT: %d\n", out, bz1->avail_in, bz1->total_in_lo32, bz1->total_out_lo32);

done:

fail:    
    free(bz1);
    free (outbuf);
    *decomprCount = byteCount;
    return ((unsigned char*)returnCharPtr);
}


/*------------------------------------------------------------------------
* CmdReturn *bztclLite_readFileCmd ()--
*    Demonstrates creating a hash entry.
*    Creates a hash entry for Key, with value String
* Arguments
*   objv[0]:	"dummy"
*   objv[1]:	"create"
*   objv[2]:	hash Key
*   objv[3]:	String
* 
* Results
*   Creates a new hash entry.  Sets error if entry already exists.
*
* Side Effects:
*   None
------------------------------------------------------------------------*/
CmdReturn *bztclLite_readFileCmd (ClientData info, 
			 Tcl_Interp *interp, 
			 int objc, 
			 Tcl_Obj * CONST *objv) {
    CmdReturn *returnStructPtr;
    char *returnCharPtr;
    Tcl_Obj *returnObjPtr;

    FILE*   f;
    BZFILE* b;

    char *outbuf, *fileName;
    int byteCount, out, bzStat, nBuf;
    
    
    byteCount = 0;

    /*
     * Print that the function was called for debugging
     */

    debugprt("bztclLite_readFileCmd called with %d args\n", objc);

    /* 
     * Allocate the space and initialize the return structure.
     */

    returnStructPtr = (CmdReturn *) malloc(sizeof (CmdReturn));
    returnStructPtr->status = TCL_OK;
    returnObjPtr    = NULL;


    fileName = Tcl_GetStringFromObj(objv[2], NULL);
    outbuf = (char *)ckalloc(10000);
    returnCharPtr = (char *)malloc(10000);

    f = fopen ( fileName, "r" );
    if (!f) {
       /* handle error */

        Tcl_AppendResult(interp, "Failed to open file \"",
                fileName, "\"", (char *) NULL);
        returnStructPtr->status = TCL_ERROR;
	goto done;
    }
    b = BZ2_bzReadOpen ( &bzStat, f, 0, 0, NULL, 0 );

    if (bzStat != BZ_OK) {
       BZ2_bzReadClose ( &bzStat, b );
       /* handle error */
        Tcl_AppendResult(interp, "BZOpen Failed to open file \"",
                fileName, "\"", (char *) NULL);
        returnStructPtr->status = TCL_ERROR;
	goto done;
    }
    
    bzStat = BZ_OK;
    debugprt("start buf: %xx\n", outbuf);

    while (bzStat == BZ_OK ) {
      nBuf = BZ2_bzRead ( &bzStat, b, outbuf, 10000 );
      if ((bzStat == BZ_OK) || (bzStat == BZ_STREAM_END)) {
         memcpy(&returnCharPtr[byteCount], outbuf, nBuf);
         byteCount += nBuf;
         returnCharPtr = (char *)realloc(returnCharPtr, byteCount + 10000);
      }
      debugprt("bytesRead: %d Total: %d buf: %xx\n", nBuf, byteCount, outbuf);
    }

    if (bzStat != BZ_STREAM_END) {
       BZ2_bzReadClose ( &bzStat, b );
       /* handle error */
        Tcl_AppendResult(interp, "BZRead Failed during processing \"",
                fileName, "\"", (char *) NULL);
        returnStructPtr->status = TCL_ERROR;
	goto done;
    } else {
       BZ2_bzReadClose ( &bzStat, b );
    }
  

done:
    if ((returnObjPtr == NULL) && (returnCharPtr != NULL)) {
        returnObjPtr = Tcl_NewStringObj(returnCharPtr, byteCount);
    }
                            
    returnStructPtr->object = returnObjPtr;
    if (returnCharPtr != NULL) {free(returnCharPtr);}

    ckfree (outbuf);
    return returnStructPtr;
}

/*------------------------------------------------------------------------
* CmdReturn *bztclLite_writeFileCmd ()--
*    Demonstrates creating a hash entry.
*    Creates a hash entry for Key, with value String
* Arguments
*   objv[0]:	"dummy"
*   objv[1]:	"create"
*   objv[2]:	hash Key
*   objv[3]:	String
* 
* Results
*   Creates a new hash entry.  Sets error if entry already exists.
*
* Side Effects:
*   None
------------------------------------------------------------------------*/
CmdReturn *bztclLite_writeFileCmd (ClientData info, 
			 Tcl_Interp *interp, 
			 int objc, 
			 Tcl_Obj * CONST *objv) {
    CmdReturn *returnStructPtr;
    Tcl_Obj *returnObjPtr;

    FILE*   f;
    BZFILE* b;

    char *outbuf, *fileName, *inbuf;
    int byteCount, out, bzStat, nBuf, outPos;
    
    
    byteCount = 0;

    /*
     * Print that the function was called for debugging
     */

    debugprt("bztclLite_writeFileCmd called with %d args\n", objc);

    /* 
     * Allocate the space and initialize the return structure.
     */

    returnStructPtr = (CmdReturn *) malloc(sizeof (CmdReturn));
    returnStructPtr->status = TCL_OK;
    returnObjPtr    = objv[2];


    fileName = Tcl_GetStringFromObj(objv[2], NULL);
    inbuf    = Tcl_GetStringFromObj(objv[3], &byteCount);
    outbuf = (char *)ckalloc(10000);


    f = fopen ( fileName, "w");

    if (!f) {
       /* handle error */
        Tcl_AppendResult(interp, "Failed to open file \"",
                fileName, "\"", (char *) NULL);
        returnStructPtr->status = TCL_ERROR;
	goto done;
    }

    b = BZ2_bzWriteOpen ( &bzStat, f, 9, 0, 30);

    if (bzStat != BZ_OK) {
       BZ2_bzWriteClose ( &bzStat, b, 1, 0,0 );
       /* handle error */
        Tcl_AppendResult(interp, "BZOpen Failed to open file \"",
                fileName, "\"", (char *) NULL);
        returnStructPtr->status = TCL_ERROR;
	goto done;
    }
    
    outPos = 0;
    while (byteCount > 0 ) {
      if (byteCount > 10000) {
        memcpy(outbuf, &inbuf[outPos], 10000);
	nBuf = 10000;
      } else {
        memcpy(outbuf, &inbuf[outPos], byteCount);
	nBuf = byteCount;
      }
      BZ2_bzWrite ( &bzStat, b, outbuf, nBuf );
      if (bzStat == BZ_OK) {
        byteCount -= nBuf;
	outPos += nBuf;
      }
      debugprt("outR1: %d \n", nBuf);
    }
    BZ2_bzWriteClose ( &bzStat, b, 0, 0, 0);

done:
    fclose(f);
                            
    returnStructPtr->object = returnObjPtr;

    ckfree (outbuf);
    return returnStructPtr;
}



/*
 *----------------------------------------------------------------------
 *
 * bztclLite_Cmd --
 *
 *      bztclLite_Cmd is invoked to process the "bztclLite" Tcl command.
 *      It will parse a subcommand, and perform the requested action.
 *   
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 * 
 *----------------------------------------------------------------------
 */

int bztclLite_Cmd(ClientData dummy, 
             Tcl_Interp *interp, 
	     int objc, 
	     Tcl_Obj * CONST *objv) {

    /* ClientData dummy;           /* Not used. */
    /* Tcl_Interp *interp;         /* Current interpreter. */
    /* int objc;                   /* Number of arguments. */
    /* Tcl_Obj *CONST objv[];      /* Argument objects. */

  int cmdnum;
  int result;
  Tcl_Obj *returnValue;
  CmdReturn *returnStruct;
  programInfo info;
  
  /*
   * Initialize the return value
   */

  returnValue = NULL;
  returnStruct = NULL;
  
  /*
   *  Check that we have at least a subcommand, 
   *   else return an Error and the usage message 
   */

  if (objc < 2) {
      Tcl_WrongNumArgs(interp, 1, objv, 
              "compress/uncompress string");
      return TCL_ERROR;
  }

  /*
   * Find this dummy subcommand in the list of subcommands.  
   * Tcl_GetIndexFromObj returns the offset of the recognized string,
   * which is used to index into the command definitions table.
   */

  result = Tcl_GetIndexFromObj(interp, objv[1], subcommands, 
  	                       "subcommand", TCL_EXACT, &cmdnum);
   
  /* 
   * If the result is not TCL_OK, then the error message is already
   *    in the Tcl Interpreter, this code can immediately return.
   */

  if (result != TCL_OK) {
      return TCL_ERROR;
  }

  /*
   *  Check that the argument count matches what's expected for this
   * Subcommand.
   */

  if ((objc < definitions[cmdnum].minArgCnt) || 
      (objc > definitions[cmdnum].maxArgCnt) ) {
      Tcl_WrongNumArgs(interp, 1, objv, definitions[cmdnum].usage);
      return TCL_ERROR;
  }
    
			
  result = TCL_OK;

  /* 
   * The subcommand is recognized, and has a valid number of arguments
   * Process the command.
   */

  switch (cmdnum) {
      case M_compress: {
		returnStruct = 
		    bztclLite_compressCmd((ClientData) &info, interp, objc, objv);
		break;
      }
      case M_uncompress: {
		returnStruct = 
		    bztclLite_uncompressCmd((ClientData) &info, interp, objc, objv);
		break;
      }
      case M_readFile: {
		returnStruct = 
		    bztclLite_readFileCmd((ClientData) &info, interp, objc, objv);
		break;
      }
      case M_writeFile: {
		returnStruct = 
		    bztclLite_writeFileCmd((ClientData) &info, interp, objc, objv);
		break;
      }
      default:		   {
      		char error[80];
		sprintf(error, "Bad sub-command %s.  Has no entry in switch",
		    Tcl_GetStringFromObj(objv[1], NULL));
		returnValue = Tcl_NewStringObj(error, -1);
		result = TCL_ERROR;
      }
  }
  
  /* 
   * Extract an object to return from returnStruc.
   * returnStruct will be NULL if the processing is done in this 
   *  function and no other function is called.
   */

  if (returnStruct != NULL) {
      returnValue = returnStruct->object;
      result = returnStruct->status;
      free (returnStruct);
  }
  
  /* 
   * Set the return value and return the status
   */

  if (returnValue != NULL) {
      Tcl_SetObjResult(interp, returnValue);
  }

  return result;
}
/* ----- C++ Comment
// This is the close of the extern "C" section
*/

/*
 }
*/




